/** @file   RedrawQueue.cpp
 * @brief   Implementation of RedrawQueue class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:40 $
 * @author  Tomi Lamminsaari
 */

#include "RedrawQueue.h"
#include <allegro.h>

namespace eng2d {


///
/// Constants, datatypes and static methods
/// ============================================================================




///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
RedrawQueue::RedrawQueue( int aLayers ) :
  iLayerCount( aLayers ),
  iQueue( aLayers )
{
  for ( int i=0; i < aLayers; i++ ) {
    iQueue.at(i) = new LayerQueue;
  }
}



/** Destructor.
 */
RedrawQueue::~RedrawQueue()
{
  this->clear();
  for ( int i=0; i < iQueue.size(); i++ ) {
    delete iQueue.at(i);
    iQueue.at(i) = 0;
  }
  iQueue.clear();
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================




///
/// New public methods
/// ============================================================================

/** Adds given object to this queue.
 */
void RedrawQueue::add( int aLayer,
                       const Vec2D& aBlitPos,
                       GfxContentType aContentType,
                       BlendMode aBlendMode,
                       int aAlpha,
                       const void* aGfx )
{
  if ( aLayer < 0 || aLayer >= iQueue.size() ) {
    return;
  }
  
  QueueItem* item = new QueueItem;
  if ( item != 0 ) {
    item->iPos = aBlitPos;
    item->iContentType = aContentType;
    item->iBlendMode = aBlendMode;
    item->iAlpha = aAlpha;
    item->iRotation = 0;
    item->iGfx = aGfx;
    iQueue.at(aLayer)->push_back( item );
  }
}



/** Adds given rotated sprite to the queue.
 */
void RedrawQueue::addRotated( int aLayer,
                              const Vec2D& aBlitPos,
                              fixed aAngle,
                              const BITMAP* aGfx )
{
  if ( aLayer < 0 || aLayer >= iQueue.size() ) {
    return;
  }
  
  QueueItem* item = new QueueItem;
  if ( item != 0 ) {
    item->iPos = aBlitPos;
    item->iContentType = ESprite;
    item->iBlendMode = ESolid;
    item->iAlpha = 0;
    item->iRotation = aAngle;
    item->iGfx = aGfx;
    iQueue.at(aLayer)->push_back( item );
  }
}



/** Draws certain layer.
 */
void RedrawQueue::redraw( int aLayer, BITMAP* aTarget ) const
{
  if ( aLayer < 0 || aLayer >= iQueue.size() || aTarget == 0 ) {
    return;
  }
  
  LayerQueue* layerData = iQueue.at( aLayer );
  for ( LayerQueue::iterator it = layerData->begin();
        it != layerData->end();
        it++ ) {
    QueueItem* item = *it;
    if ( item != 0 ) {
      switch ( item->iContentType ) {
        case ( EBitmap ): {
          this->drawBitmap( item, aTarget );
          break;
        }
        case ( ESprite ): {
          this->drawSprite( item, aTarget );
          break;
        }
        case ( ERleSprite ): {
          this->drawRleSprite( item, aTarget );
          break;
        }
        default: {
          break;
        }
      }
    }
  }
  SetBlendFunction( ESolid, 128 );
}



/** Clears the contents of this queue.
 */
void RedrawQueue::clear()
{
  // Delete old queue data.
  for ( int i=0; i < iQueue.size(); i++ ) {
    LayerQueue* layerData = iQueue.at(i);
    for ( LayerQueue::iterator it = layerData->begin();
          it != layerData->end();
          it++ ) {
      // Delete each QueueItem
      delete (*it);
      (*it) = 0;
    }
    delete layerData;
    layerData = 0;
  }
  iQueue.clear();
  
  // And reinit the data.
  for ( int i=0; i < iLayerCount; i++ ) {
    iQueue.push_back( new LayerQueue );
  }
}




///
/// Getter methods
/// ============================================================================




///
/// Protected and private methods
/// ============================================================================

/** Draws bitmap.
 */
void RedrawQueue::drawBitmap( const QueueItem* aItem, BITMAP* aTarget ) const
{
  const BITMAP* source = reinterpret_cast<const BITMAP*>( aItem->iGfx );
  blit( const_cast<BITMAP*>(source), aTarget,
        0,0, aItem->iPos.intX(), aItem->iPos.intY(),
        source->w, source->h );
}



/** Draws sprite.
 */
void RedrawQueue::drawSprite( const QueueItem* aItem, BITMAP* aTarget ) const
{
  const BITMAP* source = reinterpret_cast<const BITMAP*>( aItem->iGfx );
  if ( aItem->iRotation != 0 ) {
    // Rotation and blending cannot be used at the same time. We just
    // perform the rotation.
    rotate_sprite( aTarget, const_cast<BITMAP*>(source),
                   aItem->iPos.intX(), aItem->iPos.intY(),
                   aItem->iRotation );
    
  } else {
    if ( aItem->iBlendMode != ESolid ) {
      // Draw with blending function.
      SetBlendFunction( aItem->iBlendMode, aItem->iAlpha );
      draw_trans_sprite( aTarget, const_cast<BITMAP*>(source),
                         aItem->iPos.intX(), aItem->iPos.intY() );
      
    } else {
      draw_sprite( aTarget, const_cast<BITMAP*>(source),
                   aItem->iPos.intX(), aItem->iPos.intY() );
    }
  }
}



/** Draw rle sprite
 */
void RedrawQueue::drawRleSprite( const QueueItem* aItem, BITMAP* aTarget ) const
{
  const RLE_SPRITE* source = reinterpret_cast<const RLE_SPRITE*>( aItem->iGfx );
  if ( aItem->iBlendMode == ESolid ) {
    draw_rle_sprite( aTarget, source, aItem->iPos.intX(), aItem->iPos.intY() );
    
  } else {
    SetBlendFunction( aItem->iBlendMode, aItem->iAlpha );
    draw_trans_rle_sprite( aTarget, source, aItem->iPos.intX(), aItem->iPos.intY() );
  }
}



}
